/* -*-c++-*- OpenSceneGraph - Copyright (C) 2014 ADIT GmbH
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
*/

#include <algorithm>
#include <osgUtil/HalfFloatArrayAdapter>

namespace osgUtil {
HalfFloatArrayAdapter::HalfFloatArrayAdapter(unsigned int elements)
    : Array((elements == 2 ? Vec2hArrayType :
            (elements == 3 ? Vec3hArrayType :
            (elements == 4 ? Vec4hArrayType : HalfFloatArrayType))), elements, GL_HALF_FLOAT)
    , _data(0)
    , _size(0)
    , _elementSize(elements * sizeof(GLushort))
    , _ownsData(true) {}

HalfFloatArrayAdapter::HalfFloatArrayAdapter(unsigned int elements,
                                             unsigned int size,
                                             GLvoid *data, bool takeOwnership)
    : Array((elements == 2 ? Vec2hArrayType :
            (elements == 3 ? Vec3hArrayType :
            (elements == 4 ? Vec4hArrayType : HalfFloatArrayType))), elements, GL_HALF_FLOAT)
    , _data(static_cast<GLushort*>(data))
    , _size(size)
    , _elementSize(elements * sizeof(GLushort))
    , _ownsData(takeOwnership)
{
}

HalfFloatArrayAdapter::HalfFloatArrayAdapter(const HalfFloatArrayAdapter &other, const osg::CopyOp &copyop)
    : Array(other._arrayType, other._elementSize, GL_HALF_FLOAT)
    , _data(other._data)
    , _size(other._size)
    , _elementSize(other._elementSize)
    , _ownsData(false)
{
    if (copyop.getCopyFlags() != osg::CopyOp::SHALLOW_COPY)
    {
        _data = new GLushort[getTotalDataSize()];
        std::copy(other._data, other._data + getTotalDataSize(), _data);
        _ownsData = true;
    }
}

HalfFloatArrayAdapter::~HalfFloatArrayAdapter()
{
    if (_ownsData)
    {
        delete [] _data;
    }
}

inline int HalfFloatArrayAdapter::compare(unsigned int lhs, unsigned int rhs) const {
    switch (_arrayType) {
    case Vec2hArrayType: {
        osg::Vec2h* const data = (osg::Vec2h* const)(_data);
        if (data[lhs] < data[rhs])
            return -1;
        if (data[lhs] == data[rhs])
            return 0;
        return 1; }
    case Vec3hArrayType: {
        osg::Vec3h* const data = (osg::Vec3h* const)(_data);
        if (data[lhs] < data[rhs])
            return -1;
        if (data[lhs] == data[rhs])
            return 0;
        return 1; }
    case Vec4hArrayType: {
        osg::Vec4h* const data = (osg::Vec4h* const)(_data);
        if (data[lhs] < data[rhs])
            return -1;
        if (data[lhs] == data[rhs])
            return 0;
        return 1; }
    default:
        return 1;
    }
}

}
